<?php
session_start();
require 'db.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: admin_login.php');
    exit;
}

// Handle actions (delete, status update)
if (isset($_GET['action']) && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    
    if ($_GET['action'] == 'delete') {
        // Delete record from database
        $sql = "DELETE FROM books_data WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        
        $_SESSION['message'] = "Submission deleted successfully!";
        header('Location: admin_dashboard.php');
        exit;
    }
    
    if ($_GET['action'] == 'toggle_status') {
        // First check if status column exists, if not add it
        $check_sql = "SHOW COLUMNS FROM books_data LIKE 'status'";
        $check_result = $conn->query($check_sql);
        
        if ($check_result->num_rows == 0) {
            // Add status column if it doesn't exist
            $alter_sql = "ALTER TABLE books_data ADD COLUMN status BOOLEAN DEFAULT FALSE";
            $conn->query($alter_sql);
        }
        
        $sql = "UPDATE books_data SET status = NOT status WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        
        $_SESSION['message'] = "Status updated successfully!";
        header('Location: admin_dashboard.php');
        exit;
    }
}

// Get all submissions - ORDER BY id DESC for latest first
$sql = "SELECT * FROM books_data ORDER BY id DESC";
$result = $conn->query($sql);

// Get statistics
$stats_sql = "SELECT 
    COUNT(*) as total,
    SUM(CASE WHEN status = 1 THEN 1 ELSE 0 END) as completed,
    COUNT(*) - SUM(CASE WHEN status = 1 THEN 1 ELSE 0 END) as pending
    FROM books_data";
$stats_result = $conn->query($stats_sql);
$stats = $stats_result->fetch_assoc();

// Calculate serial numbers for display
$total_records = $result->num_rows;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Amazon Self Pub</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Raleway:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: "Raleway", sans-serif;
            background: #f8f9fa;
        }
        .sidebar {
            background: linear-gradient(135deg, #2c3e50, #34495e);
            color: white;
            min-height: 100vh;
            box-shadow: 2px 0 10px rgba(0,0,0,0.1);
        }
        .sidebar .nav-link {
            color: #ecf0f1;
            padding: 15px 20px;
            margin: 5px 0;
            border-radius: 8px;
            transition: all 0.3s ease;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
        }
        .sidebar .nav-link i {
            margin-right: 10px;
            width: 20px;
            text-align: center;
        }
        .navbar {
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            border: none;
            transition: transform 0.3s ease;
        }
        .stat-card:hover {
            transform: translateY(-5px);
        }
        .stat-card i {
            font-size: 2.5rem;
            margin-bottom: 15px;
        }
        .stat-number {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 5px;
        }
        .table-container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            overflow: hidden;
        }
        .table th {
            background: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
            font-weight: 600;
            color: #2c3e50;
        }
        .badge-pending {
            background: #ffc107;
            color: #000;
        }
        .badge-completed {
            background: #28a745;
            color: white;
        }
        .btn-action {
            padding: 5px 10px;
            margin: 2px;
            border-radius: 6px;
            border: none;
            transition: all 0.3s ease;
        }
        .btn-view {
            background: #17a2b8;
            color: white;
        }
        .btn-download {
            background: #6c63ff;
            color: white;
        }
        .btn-delete {
            background: #dc3545;
            color: white;
        }
        .btn-status {
            background: #28a745;
            color: white;
        }
        .btn-action:hover {
            transform: translateY(-2px);
            box-shadow: 0 3px 8px rgba(0,0,0,0.2);
        }
        .serial-number {
            font-weight: bold;
            color: #6c63ff;
        }
        .latest-row {
            background: linear-gradient(135deg, #e3f2fd, #f3e5f5) !important;
            border-left: 4px solid #6c63ff;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <nav class="col-md-3 col-lg-2 d-md-block sidebar collapse">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4">
                        <img src="images/logo.png" alt="Logo" style="width: 120px; margin-bottom: 20px;">
                        <h5>Amazon Self Pub</h5>
                        <small>Admin Panel</small>
                    </div>
                    
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link active" href="admin_dashboard.php">
                                <i class="fas fa-tachometer-alt"></i>
                                Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="admin_logout.php">
                                <i class="fas fa-sign-out-alt"></i>
                                Logout
                            </a>
                        </li>
                    </ul>
                </div>
            </nav>

            <!-- Main content -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <!-- Navbar -->
                <nav class="navbar navbar-expand-lg navbar-light bg-white mt-3 rounded-3">
                    <div class="container-fluid">
                        <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                            <span class="navbar-toggler-icon"></span>
                        </button>
                        <div class="collapse navbar-collapse" id="navbarNav">
                            <ul class="navbar-nav me-auto">
                                <li class="nav-item">
                                    <span class="navbar-text">
                                        Welcome, <strong><?php echo $_SESSION['admin_username']; ?></strong>
                                    </span>
                                </li>
                            </ul>
                            <span class="navbar-text">
                                <i class="fas fa-calendar"></i>
                                <?php echo date('F j, Y'); ?>
                            </span>
                        </div>
                    </div>
                </nav>

                <?php if (isset($_SESSION['message'])): ?>
                    <div class="alert alert-success alert-dismissible fade show mt-3" role="alert">
                        <?php echo $_SESSION['message']; unset($_SESSION['message']); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Statistics Cards -->
                <div class="row mt-4">
                    <div class="col-md-4 mb-4">
                        <div class="stat-card text-center">
                            <i class="fas fa-inbox text-primary"></i>
                            <div class="stat-number"><?php echo $stats['total']; ?></div>
                            <div class="text-muted">Total Submissions</div>
                        </div>
                    </div>
                </div>

                <!-- Recent Submissions -->
                <div class="table-container mt-4">
                    <div class="p-4 border-bottom">
                        <h4 class="mb-0">
                            <i class="fas fa-history me-2"></i>
                            Recent Submissions (Latest First)
                        </h4>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Book Title</th>
                                    <th>Author</th>
                                    <th>Genre</th>
                                    <th>ISBN</th>
                                    <th>Category</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $serial_number = 1;
                                while ($row = $result->fetch_assoc()): 
                                    $is_latest = $serial_number == 1; // First row is latest
                                ?>
                                <tr class="<?php echo $is_latest ? 'latest-row' : ''; ?>">
                                    <td>
                                        <span class="serial-number"><?php echo $serial_number; ?></span>
                                    </td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($row['book_title']); ?></strong>
                                        <?php if (!empty($row['tagline'])): ?>
                                            <br><small class="text-muted"><?php echo htmlspecialchars($row['tagline']); ?></small>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($row['authors_name']); ?></td>
                                    <td><?php echo htmlspecialchars($row['genre']); ?></td>
                                    <td>
                                        <code><?php echo htmlspecialchars($row['isbn']); ?></code>
                                    </td>
                                    <td>
                                        <span class="badge bg-light text-dark"><?php echo htmlspecialchars($row['design_category']); ?></span>
                                    </td>
                                    <td>
                                        <div class="btn-group">
                                            <button type="button" class="btn btn-view btn-action btn-sm" 
                                                    data-bs-toggle="modal" 
                                                    data-bs-target="#viewModal"
                                                    data-id="<?php echo $row['id']; ?>"
                                                    data-booktitle="<?php echo htmlspecialchars($row['book_title']); ?>"
                                                    data-tagline="<?php echo htmlspecialchars($row['tagline']); ?>"
                                                    data-author="<?php echo htmlspecialchars($row['authors_name']); ?>"
                                                    data-genre="<?php echo htmlspecialchars($row['genre']); ?>"
                                                    data-isbn="<?php echo htmlspecialchars($row['isbn']); ?>"
                                                    data-size="<?php echo htmlspecialchars($row['book_cover_size']); ?>"
                                                    data-backcover="<?php echo htmlspecialchars($row['back_cover_description']); ?>"
                                                    data-description="<?php echo htmlspecialchars($row['describe_your_book']); ?>"
                                                    data-category="<?php echo htmlspecialchars($row['design_category']); ?>">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                <?php 
                                $serial_number++;
                                endwhile; 
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- View Modal -->
    <div class="modal fade" id="viewModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Book Submission Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Book Title:</strong> <span id="modal-booktitle"></span></p>
                            <p><strong>Tagline:</strong> <span id="modal-tagline"></span></p>
                            <p><strong>Author:</strong> <span id="modal-author"></span></p>
                            <p><strong>Genre:</strong> <span id="modal-genre"></span></p>
                        </div>
                        <div class="col-md-6">
                            <p><strong>ISBN:</strong> <span id="modal-isbn"></span></p>
                            <p><strong>Cover Size:</strong> <span id="modal-size"></span></p>
                            <p><strong>Design Category:</strong> <span id="modal-category"></span></p>
                        </div>
                    </div>
                    
                    <div class="mt-3">
                        <strong>Back Cover Description:</strong>
                        <div class="border rounded p-3 mt-2 bg-light">
                            <span id="modal-backcover"></span>
                        </div>
                    </div>
                    
                    <div class="mt-3">
                        <strong>Book Description:</strong>
                        <div class="border rounded p-3 mt-2 bg-light">
                            <span id="modal-description"></span>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // View Modal functionality
        var viewModal = document.getElementById('viewModal');
        viewModal.addEventListener('show.bs.modal', function (event) {
            var button = event.relatedTarget;
            
            document.getElementById('modal-booktitle').textContent = button.getAttribute('data-booktitle');
            document.getElementById('modal-tagline').textContent = button.getAttribute('data-tagline') || 'N/A';
            document.getElementById('modal-author').textContent = button.getAttribute('data-author');
            document.getElementById('modal-genre').textContent = button.getAttribute('data-genre');
            document.getElementById('modal-isbn').textContent = button.getAttribute('data-isbn');
            document.getElementById('modal-size').textContent = button.getAttribute('data-size');
            document.getElementById('modal-category').textContent = button.getAttribute('data-category');
            document.getElementById('modal-backcover').textContent = button.getAttribute('data-backcover') || 'No description provided';
            document.getElementById('modal-description').textContent = button.getAttribute('data-description') || 'No description provided';
        });
    </script>
</body>
</html>