<?php
require 'db.php';
error_reporting(E_ALL);

ini_set('display_errors', 1);

// Set higher limits for large file uploads (30MB)
ini_set('upload_max_filesize', '30M');
ini_set('post_max_size', '30M');
ini_set('max_execution_time', 300);
ini_set('max_input_time', 300);
ini_set('memory_limit', '256M');

// Set custom temporary directory if needed
// ini_set('upload_tmp_dir', '/path/to/custom/tmp');

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Get form data
    $name = $conn->real_escape_string($_POST['name']);
    $email = $conn->real_escape_string($_POST['email']);
    $phone = $conn->real_escape_string($_POST['phone']);
    $genre = $conn->real_escape_string($_POST['genre']);
    $message = $conn->real_escape_string($_POST['message']);
    
    // File upload handling
    $file_name = "";
    $file_path = "";
    $file_type = "";
    $file_size = 0;
    $upload_success = false;
    
    if (isset($_FILES['file']) && $_FILES['file']['error'] == UPLOAD_ERR_OK) {
        $upload_dir = "uploads/";
        
        // Create uploads directory if it doesn't exist with proper permissions
        if (!file_exists($upload_dir)) {
            if (!mkdir($upload_dir, 0755, true)) {
                die("❌ Failed to create upload directory. Please check folder permissions.");
            }
        }
        
        // Check if directory is writable
        if (!is_writable($upload_dir)) {
            die("❌ Upload directory is not writable. Please check folder permissions.");
        }
        
        $file_tmp_name = $_FILES['file']['tmp_name'];
        $file_name = basename($_FILES['file']['name']);
        $file_type = $_FILES['file']['type'];
        $file_size = $_FILES['file']['size'];
        
        // Check if temporary file exists
        if (!file_exists($file_tmp_name)) {
            die("❌ Temporary file not found. Upload may have failed.");
        }
        
        // Check file size (25MB limit)
        if ($file_size > 25 * 1024 * 1024) {
            die("❌ File size too large. Maximum size is 25MB.");
        }
        
        // Check available disk space (need at least 50MB free)
        $free_space = disk_free_space(".");
        if ($free_space < 50 * 1024 * 1024) {
            die("❌ Insufficient disk space. Please free up some space and try again.");
        }
        
        // Validate file type
        $allowed_types = [
            'image/jpeg', 'image/jpg', 'image/png', 'image/gif',
            'application/pdf', 'application/zip', 
            'application/x-photoshop', 'image/vnd.adobe.photoshop',
            'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ];
        
        $file_extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'zip', 'psd', 'doc', 'docx'];
        
        if (!in_array($file_type, $allowed_types) || !in_array($file_extension, $allowed_extensions)) {
            die("❌ Invalid file type. Please upload images, PDFs, ZIP, PSD, or Word documents only.");
        }
        
        // Generate unique filename to prevent overwrites
        $unique_name = uniqid() . '_' . time() . '.' . $file_extension;
        $file_path = $upload_dir . $unique_name;
        
        // Move uploaded file with error handling
        if (move_uploaded_file($file_tmp_name, $file_path)) {
            $upload_success = true;
            $file_name = $conn->real_escape_string($file_name);
            $file_path = $conn->real_escape_string($file_path);
            
            // Verify the file was actually moved
            if (!file_exists($file_path)) {
                $upload_success = false;
                die("❌ File move operation failed. File not found at destination.");
            }
        } else {
            $error_message = "❌ Error moving uploaded file. ";
            
            // Get specific error information
            switch ($_FILES['file']['error']) {
                case UPLOAD_ERR_INI_SIZE:
                    $error_message .= "File exceeds server limit.";
                    break;
                case UPLOAD_ERR_FORM_SIZE:
                    $error_message .= "File exceeds form limit.";
                    break;
                case UPLOAD_ERR_PARTIAL:
                    $error_message .= "File only partially uploaded.";
                    break;
                case UPLOAD_ERR_NO_FILE:
                    $error_message .= "No file was uploaded.";
                    break;
                case UPLOAD_ERR_NO_TMP_DIR:
                    $error_message .= "Missing temporary folder.";
                    break;
                case UPLOAD_ERR_CANT_WRITE:
                    $error_message .= "Failed to write file to disk. ";
                    $error_message .= "Check folder permissions and disk space.";
                    break;
                case UPLOAD_ERR_EXTENSION:
                    $error_message .= "PHP extension stopped the file upload.";
                    break;
                default:
                    $error_message .= "Unknown error (Code: " . $_FILES['file']['error'] . ")";
                    break;
            }
            
            die($error_message);
        }
    } else {
        $file_error = $_FILES['file']['error'] ?? 'Unknown error';
        $error_messages = [
            UPLOAD_ERR_INI_SIZE => "File is too large (server limit).",
            UPLOAD_ERR_FORM_SIZE => "File is too large (form limit).",
            UPLOAD_ERR_PARTIAL => "File was only partially uploaded.",
            UPLOAD_ERR_NO_FILE => "No file was uploaded.",
            UPLOAD_ERR_NO_TMP_DIR => "Missing temporary folder.",
            UPLOAD_ERR_CANT_WRITE => "Failed to write file to disk. Check permissions.",
            UPLOAD_ERR_EXTENSION => "A PHP extension stopped the file upload."
        ];
        die("❌ File upload error: " . ($error_messages[$file_error] ?? 'Unknown error'));
    }

    // Insert into database
    $sql = "INSERT INTO book_cover_queries 
            (name, email, phone, genre, message, file_name, file_path, file_type, file_size) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("ssssssssi", $name, $email, $phone, $genre, $message, $file_name, $file_path, $file_type, $file_size);
        
        if ($stmt->execute()) {
            // Send email notification
            sendEmailNotification($name, $email, $phone, $genre, $message, $file_name, $file_type, $upload_success);
            
            // Redirect to thank you page
            header("Location: thankyou.php?name=" . urlencode($name));
            exit();
        } else {
            echo "❌ Database Error: " . $stmt->error;
            // Clean up uploaded file if database insert failed
            if ($upload_success && file_exists($file_path)) {
                unlink($file_path);
            }
        }
        
        $stmt->close();
    } else {
        echo "❌ Prepare statement failed: " . $conn->error;
        // Clean up uploaded file if prepare failed
        if ($upload_success && file_exists($file_path)) {
            unlink($file_path);
        }
    }
    
    $conn->close();
} else {
    echo "❌ Invalid request method.";
}

function sendEmailNotification($name, $email, $phone, $genre, $message, $filename, $filetype, $file_uploaded) {
    // Email configuration
    $to = "team.techcole@gmail.com"; // Admin email
    $subject = "New Book Cover Design Query - Amazon Self Pub";
    
    $file_status = $file_uploaded ? "✅ File uploaded successfully" : "❌ No file uploaded or upload failed";
    $file_type_display = getFileTypeDisplay($filetype);
    
    $email_body = "
    <!DOCTYPE html>
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; background: #f4f4f4; padding: 20px; }
            .container { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); max-width: 600px; margin: 0 auto; }
            .header { background: linear-gradient(135deg, #6c63ff, #857cff); color: white; padding: 20px; border-radius: 10px 10px 0 0; text-align: center; }
            .content { padding: 20px; }
            .field { margin-bottom: 15px; padding: 15px; background: #f9f9f9; border-radius: 8px; border-left: 4px solid #6c63ff; }
            .field-label { font-weight: bold; color: #333; display: block; margin-bottom: 5px; }
            .file-info { background: #e8f4fd; padding: 10px; border-radius: 5px; margin-top: 5px; }
            .success { color: green; }
            .error { color: red; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h2>📚 New Book Cover Design Query</h2>
                <p>Amazon Self Publishing Platform</p>
            </div>
            <div class='content'>
                <div class='field'>
                    <span class='field-label'>👤 Client Name:</span> $name
                </div>
                <div class='field'>
                    <span class='field-label'>📧 Email Address:</span> $email
                </div>
                <div class='field'>
                    <span class='field-label'>📞 Phone Number:</span> $phone
                </div>
                <div class='field'>
                    <span class='field-label'>📖 Book Genre:</span> $genre
                </div>
                <div class='field'>
                    <span class='field-label'>💭 Project Details:</span><br>".nl2br(htmlspecialchars($message))."
                </div>
                <div class='field'>
                    <span class='field-label'>📎 File Attachment:</span> 
                    <div class='file-info'>
                        <strong>File Name:</strong> $filename<br>
                        <strong>File Type:</strong> $file_type_display<br>
                        <span class='".($file_uploaded ? 'success' : 'error')."'>$file_status</span>
                    </div>
                </div>
                <div class='field'>
                    <span class='field-label'>🕒 Submission Time:</span> ".date('Y-m-d H:i:s')."
                </div>
                <div class='field' style='background: #e8f4fd; border-left-color: #2196F3;'>
                    <span class='field-label'>📋 Action Required:</span><br>
                    Please review this book cover design request and contact the client within 24 hours.
                </div>
            </div>
        </div>
    </body>
    </html>
    ";
    
    // Email headers
    $headers = "MIME-Version: 1.0" . "\r\n";
    $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
    $headers .= "From: amazonselfpub@yourdomain.com" . "\r\n";
    $headers .= "Reply-To: $email" . "\r\n";
    $headers .= "X-Priority: 1 (Highest)\r\n";
    $headers .= "X-MSMail-Priority: High\r\n";
    $headers .= "Importance: High\r\n";
    
    // Send email
    if (mail($to, $subject, $email_body, $headers)) {
        error_log("✅ Email sent successfully for: $name");
    } else {
        error_log("❌ Email failed to send for: $name");
    }
}

function getFileTypeDisplay($filetype) {
    $types = [
        'image/jpeg' => 'JPEG Image',
        'image/jpg' => 'JPEG Image',
        'image/png' => 'PNG Image',
        'image/gif' => 'GIF Image',
        'application/pdf' => 'PDF Document',
        'application/zip' => 'ZIP Archive',
        'application/x-photoshop' => 'Photoshop PSD',
        'image/vnd.adobe.photoshop' => 'Photoshop PSD',
        'application/msword' => 'Word Document',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'Word Document'
    ];
    
    return $types[$filetype] ?? $filetype;
}
?>